/*
 * MoveInfo.h
 *
 * Created 8/24/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/24/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef MOVE_INFO_H
 #define MOVE_INFO_H
 
 template <typename T> class MoveInfo;
 
 #include "global.h"
 
 #include "ActionInfo.h"
 #include "MovementInfo.h"
 #include "MovementInfoCollection.h"
 #include "pointerTo.h"
 #include "ActionTask.h"
 
 /**
  * Class specification for MoveInfo
  */
 template <typename T>
 class MoveInfo : virtual public ActionInfo<T>
 {
 // Private Static Functions
 private:
    static inline AsyncTask::DoneStatus process_move( GenericAsyncTask* task_Ptr, void* data_Ptr );
 
 // Data Members
 private:
    double _time_last_processed_move_task;
    MovementInfoCollection<T> _movement_info_collection; //_movement_distance
                                                         //_movement_duration
                                                         //_movement_direction (magnitude matters)
    double _time_move_action_was_invoked; // copy of ActionInfo::_time_action_was_invoked
    
 // Local Functions
 public:
    MoveInfo( const MovementInfoCollection<T>& movement_info_collection = MovementInfoCollection<T>() );
    MoveInfo( const MovementInfoCollection<T>& movement_info_collection, const double time_action_will_begin );
    MoveInfo( const MoveInfo<T>& move_info );
    virtual ~MoveInfo();
    inline MoveInfo<T>& operator=( const MoveInfo<T>& move_info );
    inline MovementInfoCollection<T>& get_movement_info_collection();
    inline double get_time_last_processed_move_task() const;
    //inline void set_time_last_processed_move_task( const double time );
    inline double get_time_move_action_was_invoked() const;
    //inline void set_time_move_action_was_invoked( const double time );
    virtual inline void set_time_action_was_invoked( const double time );
    virtual inline void split_tasks( ActionTask<T>* action_task_Ptr, AsyncTaskManager* task_mgr_Ptr );
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** PRIVATE STATIC FUNCTIONS **/
 
 /**
  * process_move() gradually moves the Object based on the MoveInfo contained by the specified task.
  * CAVEAT: The task pointed to by the specified task_Ptr must be an ActionTask.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (void*) data_Ptr
  * @return AsyncTask::DoneStatus
  */
 template <typename T>
 inline AsyncTask::DoneStatus MoveInfo<T>::process_move( GenericAsyncTask* task_Ptr, void* data_Ptr )
 {
    nassertr( task_Ptr != NULL, AsyncTask::DS_done );
    
    PT(ActionTask<T>) action_task_Ptr( reinterpret_cast<ActionTask<T>*>( task_Ptr ) );
    PT(MoveInfo<T>) move_info_Ptr( dynamic_cast<MoveInfo<T>*>( action_task_Ptr->get_action_info() ) );
    nassertr( move_info_Ptr != NULL, AsyncTask::DS_done );
    
    double current_time( global::_clock_Ptr->get_real_time() );
    
    // if the current time is less than the time this move action should began, 
    // then do not start processing the move action yet
    if ( current_time < move_info_Ptr->get_time_action_was_invoked() )
        return AsyncTask::DS_cont;
    
    PT(Object<T>) obj_Ptr( action_task_Ptr->get_object() );
    nassertr( obj_Ptr != NULL, AsyncTask::DS_done );
    
    MovementInfoCollection<T>& movement_info_collection( move_info_Ptr->get_movement_info_collection() );
    
    //printf( "last_process_time: %1f\n", last_process_time );
    //printf( "move_action_invoke_time: %1f\n", move_action_invoke_time );
    
    // Apply gradual movement for this Object
    
    // Notice _time_last_processed_task may be out-dated 
    // _time_last_processed_task should always be greater than or equal to _time_of_last_standard_attack
    /*if ( ralph_Ptr->_time_last_processed_task < ralph_Ptr->_time_of_last_standard_attack )
    {
        ralph_Ptr->_time_last_processed_task = ralph_Ptr->_time_of_last_standard_attack;
    }*/
    
    /*if ( last_process_time < move_action_invoke_time )
    {
        move_info_Ptr->_time_last_processed_move_task = move_action_invoke_time;
    }*/
    
    // gradually move for only the first 0.15 seconds
    /*double attack_time_movement_progress( current_time - ralph_Ptr->_time_last_processed_task );
    if ( current_time - ralph_Ptr->_time_of_last_standard_attack > 0.15 )
    {
        attack_time_movement_progress = 0.15 - ralph_Ptr->_time_last_processed_task; // + _time_of_last_standard_attack?
    }
    else if ( attack_time_movement_progress > 0.0 )
    {
        double movement_ratio( attack_time_movement_progress / 0.15 );
        ralph_Ptr->move_in_direction_of_heading( ralph_Ptr->get_h(), 3.0 * movement_ratio ); // move Object 3 units
    }*/
    // ralph_Ptr->_time_last_processed_task = current_time;
    
    //static double movement_distance_calc( 0.0 );
    
    // process the MovementInfos
    while ( movement_info_collection.size() > 0 )
    {
        PT(MovementInfo<T>) movement_info_Ptr( movement_info_collection.get_front() );
        double time_leftover_unprocessed( movement_info_Ptr->process_movement( obj_Ptr, move_info_Ptr, current_time ) );
        
        if ( time_leftover_unprocessed > 0.0 )
        {
            double move_action_invoke_time( move_info_Ptr->_time_move_action_was_invoked );
            double last_process_time( move_info_Ptr->_time_last_processed_move_task );
            nassertr( last_process_time >= move_action_invoke_time, AsyncTask::DS_done );
            
            double elapse_time_since_last_movement = movement_info_Ptr->get_duration() + move_action_invoke_time - last_process_time;
            last_process_time += elapse_time_since_last_movement;
            move_info_Ptr->_time_last_processed_move_task = last_process_time; // set the time last processed movement
            movement_info_collection.remove_front();
            
            // simulate a new task by changing the ActionTask invoke time
            move_action_invoke_time += movement_info_Ptr->get_duration();
            move_info_Ptr->_time_move_action_was_invoked = move_action_invoke_time;
        }
        else
            break; // break out of while-loop
    }
    
    if ( movement_info_collection.size() == 0 )
        return AsyncTask::DS_done;
    
    move_info_Ptr->_time_last_processed_move_task = current_time; // set the time last processed movement
    
    return AsyncTask::DS_cont;
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 MoveInfo<T>::MoveInfo( const MovementInfoCollection<T>& movement_info_collection )
             : ActionInfo<T>(),
               _time_last_processed_move_task( ActionInfo<T>::get_time_action_was_invoked() ), // -1.0
               _movement_info_collection( movement_info_collection ),
               _time_move_action_was_invoked( ActionInfo<T>::get_time_action_was_invoked() )
 {
    
 }
 
 /**
  * Alternative Constructor
  * @param (const double) time_action_will_begin - global clock time the action will be invoked
  */
 template <typename T>
 MoveInfo<T>::MoveInfo( const MovementInfoCollection<T>& movement_info_collection, const double time_action_will_begin )
             : ActionInfo<T>( time_action_will_begin ),
               _time_last_processed_move_task( ActionInfo<T>::get_time_action_was_invoked() ), // -1.0
               _movement_info_collection( movement_info_collection ),
               _time_move_action_was_invoked( ActionInfo<T>::get_time_action_was_invoked() )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 MoveInfo<T>::MoveInfo( const MoveInfo<T>& move_info )
             : ActionInfo<T>( move_info ),
               _time_last_processed_move_task( move_info._time_last_processed_move_task ),
               _movement_info_collection( move_info._movement_info_collection ),
               _time_move_action_was_invoked( move_info._time_move_action_was_invoked )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 MoveInfo<T>::~MoveInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified MoveInfo to this MoveInfo.
  *
  * @param (const MoveInfo<T>&) move_info
  * @return MoveInfo<T>&
  */
 template <typename T>
 inline MoveInfo<T>& MoveInfo<T>::operator=( const MoveInfo<T>& move_info )
 {
    ActionInfo<T>::operator=( move_info );
    _time_last_processed_move_task = move_info._time_last_processed_move_task;
    _movement_info_collection = move_info._movement_info_collection;
    _time_move_action_was_invoked = move_info._time_move_action_was_invoked;
    
    return *this;
 }
 
 /**
  * get_movement_info_collection() returns the MovementInfoCollection contained by
  * this MoveInfo.
  *
  * @return MovementInfoCollection<T>&
  */
 template <typename T>
 inline MovementInfoCollection<T>& MoveInfo<T>::get_movement_info_collection()
 {
    return _movement_info_collection;
 }
 
 /**
  * get_time_last_processed_move_task() returns the global clock time of when
  * the ActionTask, containing this MoveInfo, was processed by the function
  * it points to.
  *
  * @return double
  */
 template <typename T>
 inline double MoveInfo<T>::get_time_last_processed_move_task() const
 {
    return _time_last_processed_move_task;
 }
 
 /**
  * set_time_last_processed_move_task() sets the time of when
  * the ActionTask, containing this MoveInfo, was processed
  * to the specified time;
  *
  * @param (const double) time
  */
 //template <typename T>
 //inline void MoveInfo<T>::set_time_last_processed_move_task( const double time )
 //{
 //   _time_last_processed_move_task = time;
 //}
 
 /**
  * get_time_move_action_was_invoked() returns the global clock time of when
  * the previous MovementInfo finished processing, which is the time the current
  * MovementInfo first began.
  * CAVEAT: This is not the same as get_time_action_was_invoked().
  *
  * @return double
  */
 template <typename T>
 inline double MoveInfo<T>::get_time_move_action_was_invoked() const
 {
    return _time_move_action_was_invoked;
 }
 
 /**
  * set_time_action_was_invoked() sets the time the ActionTask was invoked to
  * the specified time.
  *
  * @param (const double) time
  */
 template <typename T>
 inline void MoveInfo<T>::set_time_action_was_invoked( const double time )
 {
    _time_move_action_was_invoked = time;
    ActionInfo<T>::set_time_action_was_invoked( time );
 }
 
 /**
  * split_tasks() either assigns the specified task to the specified 
  * task manager or splits up the task into multiple tasks before 
  * assigning the tasks to the specified task manager.
  *
  * @param (ActionTask<T>*) action_task_Ptr
  * @param (AsyncTaskManager*) task_mgr_Ptr
  */
 template <typename T>
 inline void MoveInfo<T>::split_tasks( ActionTask<T>* action_task_Ptr, AsyncTaskManager* task_mgr_Ptr )
 {
    nassertv( action_task_Ptr != NULL );
    nassertv( task_mgr_Ptr != NULL );
    
    if ( _movement_info_collection.size() != 0 )
    {
        //PT(ActionInfo<T>) move_info( new MoveInfo<T>( *reinterpret_cast<MoveInfo*>(action_task_Ptr->get_action_info()) ) );
        //task_mgr_Ptr->add( new ActionTask<T>( action_task_Ptr->get_object(), &MoveInfo<T>::move, move_info ) );
        task_mgr_Ptr->add( new ActionTask<T>( action_task_Ptr->get_object(), &MoveInfo<T>::process_move, action_task_Ptr->get_action_info() ) );
    }
 }
 
 #endif // MOVE_INFO_H